﻿using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using hive;


namespace GameNetwork
{
	public abstract class GameBaseRequest
	{
		public abstract string Path { get; }

		public abstract JSONObject Parameters { get; }
	}


	public class GameRequest : GameBaseRequest
	{
		public enum GameRequestServer
		{
			Product,
			Sandbox,
			Test,
			Dev
		}

		public delegate void GameRequestResponse (bool success, string responseString);

		public static GameRequestServer server	= GameRequestServer.Product;

		protected GameRequestResponse m_OnResponse;
	

		static string m_Method	= "POST";

		public override string Path {
			get {
				return null;
			}
		}

		public override JSONObject Parameters {
			get {
				return null;
			}
		}

		string BaseURL {
			get {
				switch (server) {
				case GameRequestServer.Product:
					return "https://stepbystep.qpyou.cn";
				case GameRequestServer.Sandbox:
					return "https://sandbox-stepbystep.qpyou.cn";
				case GameRequestServer.Test:
					return "https://test-stepbystep.qpyou.cn";
				default:
					return null;
				}
			}
		}

		public void Request (GameRequestResponse OnResponse)
		{
			m_OnResponse	= OnResponse;

			string URLString = BaseURL + Path;

			URLRequest.Instance.Request (m_Method, URLString, Parameters, URLRequestResponse);

			{
				string message = "========== Request =========="
				                + "\nURL: " + URLString + "\n";

//			foreach (var item in Parameters) {
//				if (item.Value == null)
//					continue;
//				message += "\nKey: " + item.Key.ToString() + " - Value: " + item.Value.ToString();
//			}

				message += Parameters.ToString (true);

				Debug.Log (message);
			}
		}

		void URLRequestResponse (bool isSuccess, string responseString)
		{
			{
				Debug.Log ("========== Response =========="
				+ "\nType: " + GetType ()
				+ "\nSuccess: " + isSuccess
				+ "\nResponse: " + responseString
				+ "\n============================");
			}

			if (m_OnResponse != null) {
				m_OnResponse (isSuccess, responseString);
			}
		}
	}


	/*
		@desc 계정 로그인 및 생성
 	*/
	public class RequestAccountLogin : GameRequest
	{
		// required
		public long PlayerID;
		public string AppID = "";
		public string DID = "";
		public string AuthV4Token = "";

		// optional
		public string ServerID = "";
		public string Nickname = "";

		public string Language = "";
		public string Country = "";
		public string AppVersion = "";
		public string OSVersion = "";
		public string SDKVersion = "";

		public override string Path {
			get {
				return "/api/account/login";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("player_id", PlayerID);
				json.AddField ("app_id", AppID);
				json.AddField ("did", DID);
				json.AddField ("auth_v4_token", AuthV4Token);

				// optional 
				json.AddField ("server_id", ServerID);
				json.AddField ("nickname", Nickname);
				json.AddField ("game_language", Language);
				json.AddField ("country", Country);
				json.AddField ("app_version", AppVersion);
				json.AddField ("os_version", OSVersion);
				json.AddField ("sdk_version", SDKVersion);

			
				return json;
			}
		}
	}


	/*
		@desc 캐릭터 로그인
 	*/
	public class RequestCharacterLogin : GameRequest
	{
		// required
		public string SessionToken	= "";
		public long CharacterID;
	
		// optional
		public string ServerID = "";

		public override string Path {
			get {
				return "/api/character/login";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("session_token", SessionToken);
				json.AddField ("character_id", CharacterID);

				// optional
				json.AddField ("server_id", ServerID);

				return json;
			}
		}
	}


	/*
		@desc 캐릭터 생성
 	*/
	public class RequestCharacterRegister : GameRequest
	{
		// required
		public string SessionToken	= "";

		// optional
		public string Nickname	= "";
		public string ServerID	= "";

		public override string Path {
			get {
				return "/api/character/register";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("session_token", SessionToken);

				// optional
				json.AddField ("nickname", Nickname);
				json.AddField ("server_id", ServerID);

				return json;
			}
		}
	}


	/*
		@desc 계정 아이템 소비
 	*/
	public class RequestAccountGetItems : GameRequest
	{
		//required
		public string SessionToken	= "";

		// optional
		public string ServerID = "";

		public override string Path {
			get {
				return "/api/account/get-items";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("session_token", SessionToken);

				// optional
				json.AddField ("server_id", ServerID);

				return json;
			}
		}
	}


	/*
		@desc 계정 아이템 소비
 	*/
	public class RequestAccountConsumeItem : GameRequest
	{
		// required
		public string SessionToken	= "";
		public string ItemCode = "";
		public long ItemAmount;

		// optional
		public string ServerID = "";

		public override string Path {
			get {
				return "/api/account/consume-item";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("session_token", SessionToken);
				json.AddField ("item_code", ItemCode);
				json.AddField ("item_amount", ItemAmount);

				// optional
				json.AddField ("server_id", ServerID);

				return json;
			}
		}
	}


	/*
		@desc 계정 아이템 구입
 	*/
	public class RequestAccountPurchaseItem : GameRequest
	{
		// required
		public string SessionToken	= "";
		public string ItemCode = "";
		public long ItemAmount;

		// optional
		public string ServerID = "";

		public override string Path {
			get {
				return "/api/account/purchase-item";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("session_token", SessionToken);
				json.AddField ("item_code", ItemCode);
				json.AddField ("item_amount", ItemAmount);

				// optional
				json.AddField ("server_id", ServerID);

				return json;
			}
		}
	}


	/*
		@desc 우편함 조회
 	*/
	public class RequestPostboxPosts : GameRequest
	{
		// required
		public string SessionToken = "";

		// optional
		public string ServerID = "";

		public override string Path {
			get {
				return "/api/postbox/posts";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("session_token", SessionToken);
			
				// optional
				json.AddField ("server_id", ServerID);

				return json;
			}
		}
	}


	/*
		@desc 우편 수령
 	*/
	public class RequestPostboxReceive : GameRequest
	{
		// required
		public string SessionToken = "";
		public List<int> PostIdList;

		// optional
		public string ServerID = "";

		public override string Path {
			get {
				return "/api/postbox/receive";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("session_token", SessionToken);

				JSONObject jsonObject = new JSONObject (JSONObject.Type.Array);

				if (PostIdList != null) {
					foreach (int postId in PostIdList) {
						jsonObject.Add (postId);
					}
				}

				json.AddField ("post_id_list", jsonObject);

				// optional
				json.AddField ("server_id", ServerID);

				return json;
			}
		}
	}


	/*
		@desc 스코어 조회
 	*/
	public class RequestScoreRank : GameRequest
	{
		// required
		public string SessionToken = "";

		// optional
		public string ScoreType = "";
		public string ServerID = "";

		public override string Path {
			get {
				return "/api/score/rank";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("session_token", SessionToken);

				// optional
				json.AddField ("score_type", ScoreType);
				json.AddField ("server_id", ServerID);

				return json;
			}
		}
	}


	/*
		@desc 스코어 전송
 	*/
	public class RequestScoreSend : GameRequest
	{
		// required
		public string SessionToken	= "";
		public long Score = 0L;

		// optional
		public string ScoreType = "";
		public string ServerID = "";

		public override string Path {
			get {
				return "/api/score/send";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("session_token", SessionToken);
				json.AddField ("score", Score);

				// optional
				json.AddField ("score_type", ScoreType);
				json.AddField ("server_id", ServerID);

				return json;
			}
		}
	}


	/*
		@desc 계정 정보 조회
 	*/
	public class RequestAccountInfo : GameRequest
	{
		// required
		public long PlayerID;

		public override string Path {
			get {
				return "/api/account/get-info";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("player_id", PlayerID);

				return json;
			}
		}
	}
		

	/*
		@desc 게임 정보 초기화
 	*/
	public class RequestResetAccount : GameRequest
	{
		// required
		public string sessionToken;

		public override string Path {
			get {
				return "/api/account/reset";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("session_token", sessionToken);

				return json;
			}
		}
	}


	/*
		@desc 게임 계정 삭제
 	*/
	public class RequestDeleteAccount : GameRequest
	{
		// required
		public string sessionToken;
		public long PlayerID;
		public string AppID = "";
		public string DID = "";
		public string AuthV4Token = "";

		public override string Path {
			get {
				return "/api/account/player/delete";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("session_token", sessionToken);
				json.AddField ("player_id", PlayerID);
				json.AddField ("app_id", AppID);
				json.AddField ("did", DID);
				json.AddField ("auth_v4_token", AuthV4Token);

				return json;
			}
		}
	}


	/*
		@desc 영수증 검증
 	*/
	public class RequestBillingVerify : GameRequest
	{
		// required
		public long PlayerID = 0;
		public string AppID = "";
		public int MarketID = 0;
		public String HIVEIAPReceipt = "";
		public float Price = 0;
		public string Currency = "";
		public string PurchaseBypassInfo = "";

		// optional
		public string AppVersion = "";
		public long DID = 0;
		public string HIVECountry = "";
		public string Country = "";
		public string Language = "";
		public string GameLanguage = "";
		public long UID = 0;
		public string DeviceModel = "";
		public string OSVersion = "";
		public string OSAPILevel = "";
		public string SDKVersion = "";
		public int ReceiptLevel = 0;
		public JSONObject GameInfo;

		public override string Path {
			get {
				return "/api/billing/verify";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("player_id", PlayerID);
				json.AddField ("app_id", AppID);
				json.AddField ("market_id", MarketID);
				json.AddField ("hiveiap_receipt", HIVEIAPReceipt);
				json.AddField ("price", Price);
				json.AddField ("currency", Currency);
				
				if (PurchaseBypassInfo != null) {
					json.AddField ("purchase_bypass_info", PurchaseBypassInfo);
				}

				// optional
				json.AddField ("app_version", AppVersion);
				json.AddField ("did", DID);
				json.AddField ("hive_country", HIVECountry);
				json.AddField ("country", Country);
				json.AddField ("language", Language);
				json.AddField ("game_language", GameLanguage);
				json.AddField ("uid", UID);
				json.AddField ("device_model", DeviceModel);
				json.AddField ("os_version", OSVersion);
				json.AddField ("os_api_level", OSAPILevel);
				json.AddField ("sdk_version", SDKVersion);
				json.AddField ("receipt_level", ReceiptLevel);
				json.AddField ("game_info", GameInfo);

				return json;
			}
		}
	}

	public class RequestBillingVerifySubscription : GameRequest
	{
		// required
		public long PlayerID = 0;
		public string AppID = "";
		public int MarketID = 0;
		public String HIVEIAPReceipt = "";
		public float Price = 0;
		public string Currency = "";
		public string PurchaseBypassInfo = "";

		// optional
		public string AppVersion = "";
		public long DID = 0;
		public string HIVECountry = "";
		public string Country = "";
		public string Language = "";
		public string GameLanguage = "";
		public long UID = 0;
		public string DeviceModel = "";
		public string OSVersion = "";
		public string OSAPILevel = "";
		public string SDKVersion = "";
		public int ReceiptLevel = 0;
		public JSONObject GameInfo;

		public override string Path {
			get {
				return "/api/billing/verify-subscription";
			}
		}

		public override JSONObject Parameters {
			get {
				JSONObject json = new JSONObject ();

				// required
				json.AddField ("player_id", PlayerID);
				json.AddField ("app_id", AppID);
				json.AddField ("market_id", MarketID);
				json.AddField ("hiveiap_receipt", HIVEIAPReceipt);
				json.AddField ("price", Price);
				json.AddField ("currency", Currency);
				
				if (PurchaseBypassInfo != null) {
					json.AddField ("purchase_bypass_info", PurchaseBypassInfo);
				}

				// optional
				json.AddField ("app_version", AppVersion);
				json.AddField ("did", DID);
				json.AddField ("hive_country", HIVECountry);
				json.AddField ("country", Country);
				json.AddField ("language", Language);
				json.AddField ("game_language", GameLanguage);
				json.AddField ("uid", UID);
				json.AddField ("device_model", DeviceModel);
				json.AddField ("os_version", OSVersion);
				json.AddField ("os_api_level", OSAPILevel);
				json.AddField ("sdk_version", SDKVersion);
				json.AddField ("receipt_level", ReceiptLevel);
				json.AddField ("game_info", GameInfo);

				return json;
			}
		}
	}

/*
public class GameRequestCharacterRegister : GameRequest
{
	// required
	// optional

	public override string Path {
		get {
			return "";
		}
	}

	public override Dictionary<string, object> Parameters {
		get {
			JSONObject json = new JSONObject ();

			// required
			// optional
			

			return json;
		}
	}
}
 */

 }